<?php
/**
 * Standalone Dose Sharing Report Printer
 * This file is outside the module system to avoid output buffering issues
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once __DIR__ . '/config.php';

session_start();

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    die('Access denied. Please log in.');
}

try {
    $pdo = new PDO(
        'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4',
        DB_USER,
        DB_PASS,
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION, PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC]
    );
} catch (PDOException $e) {
    die('Database connection failed: ' . htmlspecialchars($e->getMessage()));
}

$employerId = $_GET['employer_id'] ?? '';
$startDate = $_GET['start_date'] ?? '';
$endDate = $_GET['end_date'] ?? '';
$department = $_GET['department'] ?? '';
$ourEmployerName = 'RDUH';

if (!$employerId || !$startDate || !$endDate) {
    die('Missing required parameters');
}

$stmt = $pdo->prepare("SELECT employer_name, contact_name, rps_name, contact_email, contact_phone, address FROM employer_contacts WHERE id = ?");
$stmt->execute([$employerId]);
$employer = $stmt->fetch();

if (!$employer) {
    die('Employer not found');
}

$employerName = $employer['employer_name'];

// Find people
$sql = "SELECT DISTINCT p.id, p.forename, p.surname, p.ecn, d.name as dept_name
        FROM people p
        JOIN departments d ON p.department_id = d.id
        WHERE p.status = 'active' 
        AND (p.employer_2_id = ? OR p.employer_3_id = ? OR p.employer_2 = ? OR p.employer_3 = ?)";
if ($department) $sql .= " AND p.department_id = ?";
$sql .= " ORDER BY p.surname, p.forename";

$stmt = $pdo->prepare($sql);
if ($department) {
    $stmt->execute([$employerId, $employerId, $employerName, $employerName, $department]);
} else {
    $stmt->execute([$employerId, $employerId, $employerName, $employerName]);
}
$people = $stmt->fetchAll();

$reportData = [];
$totalDose = 0;
$doseCount = 0;

foreach ($people as $person) {
    $doseSql = "SELECT start_date, end_date, quantity, bodypart_label, value_mSv, employer
                FROM doses WHERE person_id = ? AND start_date >= ? AND end_date <= ? ORDER BY start_date";
    
    $doseStmt = $pdo->prepare($doseSql);
    $doseStmt->execute([$person['id'], $startDate, $endDate]);
    $allDoses = $doseStmt->fetchAll();
    
    $doses = [];
    foreach ($allDoses as $dose) {
        $doseEmployer = strtoupper(trim($dose['employer'] ?? ''));
        $ourName = strtoupper(trim($ourEmployerName));
        $externalName = strtoupper(trim($employerName));
        
        if (empty($doseEmployer) || strpos($doseEmployer, $ourName) !== false || strpos($doseEmployer, $externalName) === false) {
            $doses[] = $dose;
        }
    }
    
    $personTotal = 0;
    foreach ($doses as $dose) {
        $personTotal += floatval($dose['value_mSv']);
    }
    
    $reportData[] = [
        'name' => $person['forename'] . ' ' . $person['surname'],
        'ecn' => $person['ecn'],
        'department' => $person['dept_name'],
        'doses' => $doses,
        'total' => $personTotal
    ];
    
    $totalDose += $personTotal;
    $doseCount += count($doses);
}
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Dose Sharing Report - <?= htmlspecialchars($employerName) ?></title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: Arial, sans-serif; padding: 20px; line-height: 1.4; }
        .header { margin-bottom: 30px; padding-bottom: 20px; border-bottom: 2px solid #333; }
        .header h1 { font-size: 24px; margin-bottom: 15px; color: #333; }
        .info-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 10px; margin: 15px 0; }
        .info-item { padding: 8px 0; }
        .info-label { font-weight: bold; color: #666; font-size: 12px; }
        .info-value { color: #000; font-size: 14px; margin-top: 2px; }
        .summary-box { background: #f5f5f5; padding: 15px; margin: 20px 0; border-radius: 5px; }
        .summary-stats { display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; text-align: center; }
        .stat-value { font-size: 28px; font-weight: bold; color: #e94560; }
        .stat-label { font-size: 12px; color: #666; margin-top: 5px; }
        .person-section { margin: 30px 0; padding: 20px; border: 1px solid #ddd; border-radius: 5px; page-break-inside: avoid; }
        .person-header { margin-bottom: 15px; padding-bottom: 10px; border-bottom: 1px solid #eee; }
        .person-name { font-size: 18px; font-weight: bold; color: #333; }
        .person-details { font-size: 13px; color: #666; margin-top: 5px; }
        table { width: 100%; border-collapse: collapse; margin: 15px 0; }
        th, td { border: 1px solid #ddd; padding: 10px; text-align: left; font-size: 12px; }
        th { background: #f5f5f5; font-weight: bold; color: #333; }
        .total-row { background: #f9f9f9; font-weight: bold; }
        .no-doses { padding: 15px; background: #f9f9f9; border-radius: 5px; color: #666; font-style: italic; }
        .footer { margin-top: 30px; padding-top: 20px; border-top: 2px solid #333; font-size: 11px; color: #666; }
        .print-button { position: fixed; top: 20px; right: 20px; padding: 12px 24px; background: #e94560; color: white; border: none; border-radius: 5px; cursor: pointer; font-size: 14px; font-weight: bold; box-shadow: 0 2px 10px rgba(0,0,0,0.2); z-index: 1000; }
        .print-button:hover { background: #d63447; }
        @media print {
            .print-button { display: none; }
            body { padding: 0; }
            .person-section { page-break-inside: avoid; }
        }
    </style>
</head>
<body>
    <button class="print-button" onclick="window.print()">ЁЯЦия╕П Print / Save as PDF</button>
    
    <div class="header">
        <h1>Dose Sharing Report</h1>
        
        <div class="info-grid">
            <div>
                <div class="info-item">
                    <div class="info-label">TO:</div>
                    <div class="info-value"><?= htmlspecialchars($employerName) ?></div>
                </div>
                <?php if ($employer['contact_name']): ?>
                <div class="info-item">
                    <div class="info-label">ATTENTION:</div>
                    <div class="info-value"><?= htmlspecialchars($employer['contact_name']) ?></div>
                </div>
                <?php endif; ?>
                <?php if ($employer['rps_name']): ?>
                <div class="info-item">
                    <div class="info-label">RPS:</div>
                    <div class="info-value"><?= htmlspecialchars($employer['rps_name']) ?></div>
                </div>
                <?php endif; ?>
                <?php if ($employer['contact_email']): ?>
                <div class="info-item">
                    <div class="info-label">EMAIL:</div>
                    <div class="info-value"><?= htmlspecialchars($employer['contact_email']) ?></div>
                </div>
                <?php endif; ?>
            </div>
            
            <div>
                <div class="info-item">
                    <div class="info-label">FROM:</div>
                    <div class="info-value"><?= htmlspecialchars($ourEmployerName) ?></div>
                </div>
                <div class="info-item">
                    <div class="info-label">PERIOD:</div>
                    <div class="info-value"><?= date('d/m/Y', strtotime($startDate)) ?> to <?= date('d/m/Y', strtotime($endDate)) ?></div>
                </div>
                <div class="info-item">
                    <div class="info-label">GENERATED:</div>
                    <div class="info-value"><?= date('d/m/Y H:i') ?></div>
                </div>
                <div class="info-item">
                    <div class="info-label">GENERATED BY:</div>
                    <div class="info-value"><?= htmlspecialchars($_SESSION['username']) ?></div>
                </div>
            </div>
        </div>
    </div>
    
    <div class="summary-box">
        <div class="summary-stats">
            <div>
                <div class="stat-value"><?= count($reportData) ?></div>
                <div class="stat-label">Shared Employees</div>
            </div>
            <div>
                <div class="stat-value"><?= $doseCount ?></div>
                <div class="stat-label">Dose Records</div>
            </div>
            <div>
                <div class="stat-value"><?= number_format($totalDose, 2) ?></div>
                <div class="stat-label">Total Dose (mSv)</div>
            </div>
        </div>
    </div>
    
    <p style="margin: 20px 0; padding: 10px; background: #fff3cd; border-radius: 5px; font-size: 13px;">
        <strong>Note:</strong> This report contains dose information from <strong><?= htmlspecialchars($ourEmployerName) ?></strong> only for employees who work for both organizations.
    </p>
    
    <?php if (empty($reportData)): ?>
        <div style="padding: 30px; text-align: center; background: #f9f9f9; border-radius: 5px;">
            <p style="font-size: 16px; color: #666;">No shared employees found for this period.</p>
        </div>
    <?php else: ?>
        <?php foreach ($reportData as $person): ?>
        <div class="person-section">
            <div class="person-header">
                <div class="person-name"><?= htmlspecialchars($person['name']) ?></div>
                <div class="person-details">
                    ECN: <?= htmlspecialchars($person['ecn']) ?: 'N/A' ?> | 
                    Department: <?= htmlspecialchars($person['department']) ?>
                </div>
            </div>
            
            <?php if (empty($person['doses'])): ?>
                <div class="no-doses">
                    No doses recorded at <?= htmlspecialchars($ourEmployerName) ?> for this period
                </div>
            <?php else: ?>
                <table>
                    <thead>
                        <tr>
                            <th>Wear Period</th>
                            <th>Quantity</th>
                            <th>Body Part</th>
                            <th style="text-align: right;">Value (mSv)</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($person['doses'] as $dose): ?>
                        <tr>
                            <td><?= date('d/m/Y', strtotime($dose['start_date'])) ?> - <?= date('d/m/Y', strtotime($dose['end_date'])) ?></td>
                            <td><?= htmlspecialchars($dose['quantity']) ?></td>
                            <td><?= htmlspecialchars($dose['bodypart_label']) ?></td>
                            <td style="text-align: right;"><?= number_format($dose['value_mSv'], 4) ?></td>
                        </tr>
                        <?php endforeach; ?>
                        <tr class="total-row">
                            <td colspan="3"><strong>Total for <?= htmlspecialchars($person['name']) ?></strong></td>
                            <td style="text-align: right;"><strong><?= number_format($person['total'], 4) ?> mSv</strong></td>
                        </tr>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        <?php endforeach; ?>
    <?php endif; ?>
    
    <div class="footer">
        <p><strong>Disclaimer:</strong> This report contains dose information from <?= htmlspecialchars($ourEmployerName) ?> only. External employer doses should be tracked separately by the respective organizations.</p>
        <p style="margin-top: 10px;">Report Reference: PDMS-DSR-<?= date('Ymd-His') ?></p>
    </div>
</body>
</html>