<?php
/**
 * Merge Queue Page - Review and resolve ambiguous import records
 * Includes external employer assignment workflow
 */

// Handle merge acceptance
if ($action === 'accept' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $rowId = $_POST['row_id'];
    $personId = $_POST['person_id'];
    
    $stmt = $pdo->prepare("SELECT * FROM import_rows WHERE id = ?");
    $stmt->execute([$rowId]);
    $row = $stmt->fetch();
    
    if (!$row) {
        redirect('?page=merges', 'Record not found', 'error');
    }
    
    $data = json_decode($row['raw_json'], true);
    
    // Create merge rule
    $mergeKey = generateMergeKey($data['Forename'], $data['Surname'], $data['Department'], $data['ECN']);
    $pdo->prepare("INSERT INTO merge_rules (key_hash, person_id, accepted_by, accepted_at) 
                   VALUES (?, ?, ?, NOW()) ON DUPLICATE KEY UPDATE person_id = VALUES(person_id)")
        ->execute([$mergeKey, $personId, $_SESSION['user_id']]);
    
    // Update row status
    $pdo->prepare("UPDATE import_rows SET status = 'ok', person_match_id = ? WHERE id = ?")
        ->execute([$personId, $rowId]);
    
    audit($pdo, 'merge_accept', 'merge_rules', $pdo->lastInsertId(), [
        'row_id' => $rowId,
        'person_id' => $personId
    ]);
    
    redirect('?page=merges', 'Merge accepted successfully');
}

// Handle creating new person from merge
elseif ($action === 'create' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $rowId = $_POST['row_id'];
    $deptId = $_POST['department_id'];
    
    $stmt = $pdo->prepare("SELECT * FROM import_rows WHERE id = ?");
    $stmt->execute([$rowId]);
    $row = $stmt->fetch();
    
    if (!$row) {
        redirect('?page=merges', 'Record not found', 'error');
    }
    
    $data = json_decode($row['raw_json'], true);
    
    // Create new person
    $pdo->prepare("INSERT INTO people (department_id, forename, surname, ecn, status) 
                   VALUES (?, ?, ?, ?, 'active')")
        ->execute([$deptId, $data['Forename'], $data['Surname'], $data['ECN']]);
    
    $newPersonId = $pdo->lastInsertId();
    
    // Create merge rule
    $mergeKey = generateMergeKey($data['Forename'], $data['Surname'], $data['Department'], $data['ECN']);
    $pdo->prepare("INSERT INTO merge_rules (key_hash, person_id, accepted_by, accepted_at) 
                   VALUES (?, ?, ?, NOW())")
        ->execute([$mergeKey, $newPersonId, $_SESSION['user_id']]);
    
    // Update row status
    $pdo->prepare("UPDATE import_rows SET status = 'ok', person_match_id = ? WHERE id = ?")
        ->execute([$newPersonId, $rowId]);
    
    audit($pdo, 'merge_create_person', 'people', $newPersonId, ['from_merge' => $rowId]);
    
    redirect('?page=merges', 'Person created and linked successfully');
}

// Handle ignoring a merge row
elseif ($action === 'ignore') {
    $rowId = $_GET['id'];
    
    $pdo->prepare("UPDATE import_rows SET status = 'error', message = 'Manually ignored' WHERE id = ?")
        ->execute([$rowId]);
    
    redirect('?page=merges', 'Record ignored');
}

// Handle external employer assignment
elseif ($action === 'assign_employer' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $rowId = $_POST['row_id'];
    $personId = $_POST['person_id'];
    $employerName = $_POST['employer_name'];
    $employerId = $_POST['employer_id'] ?? null;
    $employerSlot = $_POST['employer_slot'];
    
    $pdo->beginTransaction();
    
    try {
        // Get or create employer contact
        if (!$employerId) {
            $pdo->prepare("INSERT INTO employer_contacts (employer_name, is_active) VALUES (?, 1)")
                ->execute([$employerName]);
            $employerId = $pdo->lastInsertId();
        }
        
        // Assign employer to person
        $field = "employer_{$employerSlot}_id";
        $pdo->prepare("UPDATE people SET $field = ? WHERE id = ?")
            ->execute([$employerId, $personId]);
        
        // Get import row data to create dose
        $stmt = $pdo->prepare("SELECT ir.*, i.id as import_id FROM import_rows ir 
                              JOIN imports i ON ir.import_id = i.id WHERE ir.id = ?");
        $stmt->execute([$rowId]);
        $row = $stmt->fetch();
        
        $data = json_decode($row['raw_json'], true);
        
        // Get template for this import
        $importId = $row['import_id'];
        $templateStmt = $pdo->prepare("SELECT t.* FROM imports i 
                                       JOIN templates t ON i.provider = t.provider 
                                       WHERE i.id = ? LIMIT 1");
        $templateStmt->execute([$importId]);
        $template = $templateStmt->fetch();
        
        if ($template) {
            $unitMap = json_decode($template['unit_map'], true);
            $bodypartMap = json_decode($template['bodypart_map'], true);
            
            $startDate = parseDate($data['PeriodStart']);
            $endDate = parseDate($data['PeriodEnd']);
            $valueMsv = normalizeUnit($data['Value'], $data['Unit'], $unitMap);
            list($quantity, $bodypartLabel) = deriveQuantityFromBodypart($data['BadgeType'], $bodypartMap);
            
            // Insert dose
            $doseStmt = $pdo->prepare("INSERT INTO doses (person_id, start_date, end_date, quantity, bodypart_label, value_mSv, unit_raw, employer, source, badge_serial, created_by)
                                       VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'import', ?, ?)
                                       ON DUPLICATE KEY UPDATE value_mSv = VALUES(value_mSv)");
            $doseStmt->execute([
                $personId, $startDate, $endDate, $quantity, $bodypartLabel,
                $valueMsv, $data['Unit'], $employerName, $data['ECN'], $_SESSION['user_id']
            ]);
            
            // Update row status
            $pdo->prepare("UPDATE import_rows SET status = 'ok', person_match_id = ? WHERE id = ?")
                ->execute([$personId, $rowId]);
            
            $pdo->commit();
            
            audit($pdo, 'assign_employer_from_merge', 'people', $personId, [
                'employer' => $employerName,
                'slot' => $employerSlot,
                'row_id' => $rowId
            ]);
            
            redirect('?page=merges', "Employer '$employerName' assigned to slot $employerSlot and dose imported successfully");
        } else {
            throw new Exception("Could not find import template to process dose data");
        }
        
    } catch (Exception $e) {
        $pdo->rollBack();
        redirect('?page=merges', 'Error: ' . $e->getMessage(), 'error');
    }
}

// Main merge queue display
else {
    $mergeQueue = $pdo->query("SELECT ir.*, i.provider, i.filename 
                               FROM import_rows ir 
                               JOIN imports i ON ir.import_id = i.id 
                               WHERE ir.status IN ('needs_merge', 'needs_employer_assignment') 
                               ORDER BY ir.created_at DESC")->fetchAll();
    
    $departments = $pdo->query("SELECT * FROM departments ORDER BY name")->fetchAll();
    $employerContacts = $pdo->query("SELECT * FROM employer_contacts WHERE is_active = 1 ORDER BY employer_name")->fetchAll();
    
    echo "<h1>Merge Review Queue</h1>";
    
    if (empty($mergeQueue)) {
        echo "<div class='card'>";
        echo "<div style='text-align:center; padding:40px;'>";
        echo "<div style='font-size:48px; color:var(--success); margin-bottom:20px;'>✓</div>";
        echo "<h2>All Clear!</h2>";
        echo "<p>No pending merges or reviews needed. Great job!</p>";
        echo "</div>";
        echo "</div>";
    } else {
        echo "<div class='card'>";
        echo "<p><strong>" . count($mergeQueue) . "</strong> records need review</p>";
        echo "</div>";
        
        foreach ($mergeQueue as $item) {
            $data = json_decode($item['raw_json'], true);
            $isEmployerIssue = ($item['status'] === 'needs_employer_assignment');
            
            echo "<div class='card' style='border-left: 4px solid var(--warning);'>";
            
            echo "<h3>" . esc($data['Forename'] . ' ' . $data['Surname']) . "</h3>";
            
            echo "<div style='display:grid; grid-template-columns:repeat(auto-fit,minmax(200px,1fr)); gap:15px; margin:15px 0;'>";
            echo "<div><strong>Department:</strong><br>" . esc($data['Department']) . "</div>";
            echo "<div><strong>ECN:</strong><br>" . esc($data['ECN']) . "</div>";
            
            if (!empty($data['Employer'])) {
                echo "<div><strong style='color:var(--error)'>External Employer:</strong><br>" . esc($data['Employer']) . "</div>";
            }
            
            echo "<div><strong>Period:</strong><br>" . esc($data['PeriodStart']) . " to " . esc($data['PeriodEnd']) . "</div>";
            echo "<div><strong>Dose:</strong><br>" . esc($data['Value']) . " " . esc($data['Unit']) . "</div>";
            echo "</div>";
            
            echo "<div class='alert alert-warning'>";
            echo "<strong>Issue:</strong> " . esc($item['message']);
            echo "</div>";
            
            // External employer assignment workflow
            if ($isEmployerIssue && $item['person_match_id']) {
                $personStmt = $pdo->prepare("SELECT p.*, d.name as dept_name, 
                                             e2.employer_name as emp2, e3.employer_name as emp3,
                                             p.employer_2_id, p.employer_3_id
                                             FROM people p 
                                             JOIN departments d ON p.department_id = d.id
                                             LEFT JOIN employer_contacts e2 ON p.employer_2_id = e2.id
                                             LEFT JOIN employer_contacts e3 ON p.employer_3_id = e3.id
                                             WHERE p.id = ?");
                $personStmt->execute([$item['person_match_id']]);
                $matchedPerson = $personStmt->fetch();
                
                if ($matchedPerson) {
                    echo "<div style='margin-top:20px; padding-top:20px; border-top:1px solid var(--border);'>";
                    echo "<h4>External Employer Assignment</h4>";
                    
                    echo "<div style='background:var(--bg-card); padding:15px; border-radius:5px; margin:15px 0;'>";
                    echo "<strong>Matched Person:</strong> " . esc($matchedPerson['forename'] . ' ' . $matchedPerson['surname']) . "<br>";
                    echo "<strong>Department:</strong> " . esc($matchedPerson['dept_name']) . "<br>";
                    echo "<strong>External Employer to Assign:</strong> <span style='color:var(--error)'>" . esc($data['Employer']) . "</span>";
                    echo "</div>";
                    
                    // Check available slots
                    $availableSlots = [];
                    if (empty($matchedPerson['employer_2_id'])) $availableSlots[] = '2';
                    if (empty($matchedPerson['employer_3_id'])) $availableSlots[] = '3';
                    
                    if (empty($availableSlots)) {
                        echo "<div class='alert alert-error'>";
                        echo "⚠️ This person already has 2 external employers assigned. Cannot add more.";
                        echo "</div>";
                    } else {
                        echo "<form method='post' action='?page=merges&action=assign_employer' style='margin-top:15px;'>";
                        echo "<input type='hidden' name='row_id' value='{$item['id']}'>";
                        echo "<input type='hidden' name='person_id' value='{$matchedPerson['id']}'>";
                        echo "<input type='hidden' name='employer_name' value='" . esc($data['Employer']) . "'>";
                        
                        // Check if employer exists
                        $empFound = null;
                        foreach ($employerContacts as $emp) {
                            if (strcasecmp($emp['employer_name'], $data['Employer']) === 0) {
                                $empFound = $emp;
                                break;
                            }
                        }
                        
                        if ($empFound) {
                            echo "<div class='alert alert-info'>";
                            echo "✓ Employer found in system: <strong>" . esc($empFound['employer_name']) . "</strong>";
                            echo "</div>";
                            echo "<input type='hidden' name='employer_id' value='{$empFound['id']}'>";
                        } else {
                            echo "<div class='alert alert-warning'>";
                            echo "⚠️ Employer not in system. It will be created automatically.";
                            echo "</div>";
                        }
                        
                        // Slot selection
                        if (count($availableSlots) > 1) {
                            echo "<div class='form-group'>";
                            echo "<label>Assign as:</label>";
                            echo "<select name='employer_slot' required>";
                            foreach ($availableSlots as $slot) {
                                echo "<option value='$slot'>Employer $slot</option>";
                            }
                            echo "</select>";
                            echo "</div>";
                        } else {
                            echo "<input type='hidden' name='employer_slot' value='{$availableSlots[0]}'>";
                            echo "<p><strong>Will be assigned as:</strong> Employer {$availableSlots[0]}</p>";
                        }
                        
                        echo "<div style='margin-top:15px;'>";
                        echo "<button type='submit' class='btn btn-primary'>Assign Employer & Import Dose</button>";
                        echo "</div>";
                        echo "</form>";
                    }
                    
                    echo "</div>";
                }
            }
            
            // Standard merge workflow
            else {
                echo "<div style='margin-top:20px; padding-top:20px; border-top:1px solid var(--border);'>";
                echo "<h4>Resolution Options</h4>";
                
                // Find possible matches
                $stmt = $pdo->prepare("SELECT p.*, d.name as dept_name FROM people p 
                                       JOIN departments d ON p.department_id = d.id 
                                       WHERE LOWER(p.forename) = ? AND LOWER(p.surname) = ? 
                                       AND p.status = 'active'
                                       ORDER BY p.department_id");
                $stmt->execute([strtolower($data['Forename']), strtolower($data['Surname'])]);
                $matches = $stmt->fetchAll();
                
                if (!empty($matches)) {
                    echo "<h5>Link to Existing Person:</h5>";
                    echo "<form method='post' action='?page=merges&action=accept'>";
                    echo "<input type='hidden' name='row_id' value='{$item['id']}'>";
                    
                    echo "<div style='display:grid; gap:10px; margin:15px 0;'>";
                    foreach ($matches as $m) {
                        echo "<label style='padding:15px; border:1px solid var(--border); border-radius:5px; cursor:pointer; display:flex; align-items:center;'>";
                        echo "<input type='radio' name='person_id' value='{$m['id']}' required style='width:auto; margin-right:10px;'>";
                        echo "<div>";
                        echo "<strong>" . esc($m['forename'] . ' ' . $m['surname']) . "</strong><br>";
                        echo "<small style='color:var(--text-secondary)'>";
                        echo "Dept: " . esc($m['dept_name']) . " | ECN: " . esc($m['ecn']) . " | Status: " . esc($m['status']);
                        echo "</small>";
                        echo "</div>";
                        echo "</label>";
                    }
                    echo "</div>";
                    
                    echo "<button type='submit' class='btn btn-primary'>Link to Selected Person</button>";
                    echo "</form>";
                } else {
                    echo "<p style='color:var(--text-secondary)'>No existing people found with this name.</p>";
                }
                
                // Create new person option
                echo "<h5 style='margin-top:25px;'>Or Create New Person:</h5>";
                echo "<form method='post' action='?page=merges&action=create' style='display:flex; gap:10px; align-items:end;'>";
                echo "<input type='hidden' name='row_id' value='{$item['id']}'>";
                
                echo "<div class='form-group' style='margin:0; flex:1;'>";
                echo "<label>Department</label>";
                echo "<select name='department_id' required>";
                foreach ($departments as $d) {
                    $selected = (strcasecmp($d['name'], $data['Department']) === 0) ? ' selected' : '';
                    echo "<option value='{$d['id']}'$selected>" . esc($d['name']) . "</option>";
                }
                echo "</select>";
                echo "</div>";
                
                echo "<button type='submit' class='btn btn-success'>Create New Person</button>";
                echo "</form>";
                
                echo "</div>";
            }
            
            // Ignore option
            echo "<div style='margin-top:20px; padding-top:20px; border-top:1px solid var(--border);'>";
            echo "<a href='?page=merges&action=ignore&id={$item['id']}' class='btn btn-secondary btn-sm' onclick='return confirm(\"Ignore this record?\")'>Ignore This Record</a>";
            echo "</div>";
            
            echo "</div>";
        }
    }
}
