<?php
/**
 * Import Page - Handles CSV/Excel dose data imports
 * Includes proper external employer detection and merge queue handling
 */

// Handle file upload and preview
if ($action === 'upload' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $templateId = $_POST['template_id'];
    
    $stmt = $pdo->prepare("SELECT * FROM templates WHERE id = ?");
    $stmt->execute([$templateId]);
    $template = $stmt->fetch();
    
    if (!$template || !isset($_FILES['file'])) {
        redirect('?page=import', 'Invalid upload', 'error');
    }
    
    $file = $_FILES['file'];
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $rows = [];
    
    // Parse file based on type
    if ($ext === 'csv' || $ext === 'tsv') {
        $delimiter = $ext === 'tsv' ? "\t" : ',';
        $handle = fopen($file['tmp_name'], 'r');
        
        $headers = $template['titles_in_first_row'] ? fgetcsv($handle, 0, $delimiter) : [];
        
        while (($row = fgetcsv($handle, 0, $delimiter)) !== false) {
            if ($template['titles_in_first_row'] && $headers) {
                $rows[] = array_combine($headers, $row);
            } else {
                $rows[] = $row;
            }
        }
        fclose($handle);
    } else {
        redirect('?page=import', 'Unsupported file type. Please use CSV or TSV format.', 'error');
    }
    
    $_SESSION['import_preview'] = [
        'template' => $template,
        'rows' => $rows,
        'filename' => $file['name']
    ];
    
    redirect('?page=import&action=preview');
}

// Preview imported data
elseif ($action === 'preview') {
    if (!isset($_SESSION['import_preview'])) {
        redirect('?page=import');
    }
    
    $preview = $_SESSION['import_preview'];
    $template = $preview['template'];
    $rows = array_slice($preview['rows'], 0, 10);
    
    echo "<h1>Import Preview</h1>";
    
    echo "<div class='card'>";
    echo "<h2>File Information</h2>";
    echo "<p><strong>File:</strong> " . esc($preview['filename']) . "</p>";
    echo "<p><strong>Template:</strong> " . esc($template['name']) . "</p>";
    echo "<p><strong>Total Rows:</strong> " . count($preview['rows']) . "</p>";
    echo "</div>";
    
    echo "<div class='card'>";
    echo "<h2>Preview (First 10 Rows)</h2>";
    echo "<div style='overflow-x: auto;'>";
    echo "<table>";
    echo "<thead><tr>";
    
    $colMap = json_decode($template['column_map'], true);
    foreach ($colMap as $field => $col) {
        echo "<th>" . esc($field) . "</th>";
    }
    echo "</tr></thead>";
    echo "<tbody>";
    
    foreach ($rows as $row) {
        echo "<tr>";
        foreach ($colMap as $field => $col) {
            echo "<td>" . esc($row[$col] ?? '') . "</td>";
        }
        echo "</tr>";
    }
    
    echo "</tbody></table>";
    echo "</div></div>";
    
    echo "<div class='card'>";
    echo "<div class='flex-between'>";
    echo "<a href='?page=import' class='btn btn-secondary'>Cancel</a>";
    echo "<form method='post' action='?page=import&action=commit' style='display:inline;'>";
    echo "<button type='submit' class='btn btn-primary'>Commit Import</button>";
    echo "</form>";
    echo "</div></div>";
}

// Commit the import
elseif ($action === 'commit' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_SESSION['import_preview'])) {
        redirect('?page=import');
    }
    
    $preview = $_SESSION['import_preview'];
    $template = $preview['template'];
    $rows = $preview['rows'];
    
    $colMap = json_decode($template['column_map'], true);
    $unitMap = json_decode($template['unit_map'], true);
    $bodypartMap = json_decode($template['bodypart_map'], true);
    
    // Create import record
    $stmt = $pdo->prepare("INSERT INTO imports (provider, filename, row_count, status, created_by) VALUES (?, ?, ?, 'committed', ?)");
    $stmt->execute([$template['provider'], $preview['filename'], count($rows), $_SESSION['user_id']]);
    $importId = $pdo->lastInsertId();
    
    $stats = [
        'added' => 0,
        'merge_needed' => 0,
        'errors' => 0,
        'new_people' => 0,
        'zero_doses' => 0,
        'external_employer_review' => 0
    ];
    
    $pdo->beginTransaction();
    
    try {
        foreach ($rows as $row) {
            try {
                $data = [];
                foreach ($colMap as $field => $col) {
                    $data[$field] = $row[$col] ?? '';
                }
                
                $forename = trim($data['Forename']);
                $surname = trim($data['Surname']);
                $ecn = trim($data['ECN']);
                $deptName = trim($data['Department']);
                $startDate = parseDate($data['PeriodStart']);
                $endDate = parseDate($data['PeriodEnd']);
                $valueMsv = normalizeUnit($data['Value'], $data['Unit'], $unitMap);
                $doseEmployer = trim($data['Employer'] ?? '');
                
                list($quantity, $bodypartLabel) = deriveQuantityFromBodypart($data['BadgeType'], $bodypartMap);
                
                if (!$startDate || !$endDate) {
                    $stats['errors']++;
                    continue;
                }
                
                if ($valueMsv == 0) {
                    $stats['zero_doses']++;
                }
                
                // Determine if this is an external employer dose
                $isExternalDose = false;
                if (!empty($doseEmployer)) {
                    // Check if employer matches a department
                    $checkDept = $pdo->prepare("SELECT id FROM departments WHERE name = ?");
                    $checkDept->execute([$doseEmployer]);
                    $isDeptMatch = $checkDept->fetch();
                    
                    if (!$isDeptMatch) {
                        $isExternalDose = true;
                    }
                }
                
                // Get or create department
                if (empty($deptName)) {
                    if ($isExternalDose) {
                        $deptName = "External - " . $doseEmployer;
                    } else {
                        $deptName = "Unknown";
                    }
                }
                
                $stmt = $pdo->prepare("SELECT id FROM departments WHERE name = ?");
                $stmt->execute([$deptName]);
                $dept = $stmt->fetch();
                
                if (!$dept) {
                    $stmt = $pdo->prepare("INSERT INTO departments (name, is_active) VALUES (?, 1)");
                    $stmt->execute([$deptName]);
                    $deptId = $pdo->lastInsertId();
                } else {
                    $deptId = $dept['id'];
                }
                
                // Find matching person
                $stmt = $pdo->prepare("SELECT id, ecn, employer_2_id, employer_3_id FROM people 
                                       WHERE department_id = ? AND LOWER(forename) = ? AND LOWER(surname) = ? 
                                       AND status = 'active'");
                $stmt->execute([$deptId, strtolower($forename), strtolower($surname)]);
                $matches = $stmt->fetchAll();
                
                $personId = null;
                
                if (count($matches) === 1) {
                    $match = $matches[0];
                    $matchEcn = trim($match['ecn']);
                    $importEcn = trim($ecn);
                    
                    // Check ECN compatibility
                    $ecnCompatible = false;
                    if (empty($matchEcn) && empty($importEcn)) {
                        $ecnCompatible = true;
                    } elseif (!empty($matchEcn) && !empty($importEcn) && $matchEcn === $importEcn) {
                        $ecnCompatible = true;
                    } elseif (empty($matchEcn) || empty($importEcn)) {
                        $ecnCompatible = true;
                    }
                    
                    if ($ecnCompatible) {
                        // Check if external employer is assigned
                        if ($isExternalDose) {
                            $empCheck = $pdo->prepare("SELECT e2.employer_name as emp2, e3.employer_name as emp3 
                                                       FROM people p
                                                       LEFT JOIN employer_contacts e2 ON p.employer_2_id = e2.id
                                                       LEFT JOIN employer_contacts e3 ON p.employer_3_id = e3.id
                                                       WHERE p.id = ?");
                            $empCheck->execute([$match['id']]);
                            $personEmp = $empCheck->fetch();
                            
                            $hasEmployer = false;
                            if (!empty($personEmp['emp2']) && strcasecmp($personEmp['emp2'], $doseEmployer) === 0) {
                                $hasEmployer = true;
                            }
                            if (!empty($personEmp['emp3']) && strcasecmp($personEmp['emp3'], $doseEmployer) === 0) {
                                $hasEmployer = true;
                            }
                            
                            if (!$hasEmployer) {
                                // Flag for employer assignment
                                $stmt = $pdo->prepare("INSERT INTO import_rows (import_id, raw_json, status, message, person_match_id) 
                                                       VALUES (?, ?, 'needs_employer_assignment', ?, ?)");
                                $message = "External employer '$doseEmployer' needs to be assigned to $forename $surname";
                                $stmt->execute([$importId, json_encode($data), $message, $match['id']]);
                                $stats['external_employer_review']++;
                                continue;
                            }
                        }
                        
                        $personId = $match['id'];
                        
                        // Update ECN if needed
                        if (empty($matchEcn) && !empty($importEcn)) {
                            $pdo->prepare("UPDATE people SET ecn = ? WHERE id = ?")->execute([$importEcn, $match['id']]);
                        }
                    }
                } elseif (count($matches) === 0) {
                    // Check merge rules
                    $mergeKey = generateMergeKey($forename, $surname, $deptName, $ecn);
                    $stmt = $pdo->prepare("SELECT person_id FROM merge_rules WHERE key_hash = ?");
                    $stmt->execute([$mergeKey]);
                    $rule = $stmt->fetch();
                    
                    if ($rule) {
                        $personId = $rule['person_id'];
                    } else {
                        // Check if same name exists in other departments
                        $stmt = $pdo->prepare("SELECT COUNT(*) FROM people 
                                               WHERE LOWER(forename) = ? AND LOWER(surname) = ? 
                                               AND status = 'active' AND department_id != ?");
                        $stmt->execute([strtolower($forename), strtolower($surname), $deptId]);
                        $otherDeptCount = $stmt->fetchColumn();
                        
                        if ($otherDeptCount > 0 && !$isExternalDose) {
                            // Flag for merge review
                            $stmt = $pdo->prepare("INSERT INTO import_rows (import_id, raw_json, status, message) 
                                                   VALUES (?, ?, 'needs_merge', ?)");
                            $stmt->execute([$importId, json_encode($data), "Person '$forename $surname' exists in other departments"]);
                            $stats['merge_needed']++;
                            continue;
                        } else {
                            // Create new person
                            $stmt = $pdo->prepare("INSERT INTO people (department_id, forename, surname, ecn, status) 
                                                   VALUES (?, ?, ?, ?, 'active')");
                            $stmt->execute([$deptId, $forename, $surname, $ecn]);
                            $personId = $pdo->lastInsertId();
                            $stats['new_people']++;
                        }
                    }
                } else {
                    // Multiple matches - flag for review
                    $stmt = $pdo->prepare("INSERT INTO import_rows (import_id, raw_json, status, message) 
                                           VALUES (?, ?, 'needs_merge', ?)");
                    $stmt->execute([$importId, json_encode($data), "Multiple matches for '$forename $surname' in $deptName"]);
                    $stats['merge_needed']++;
                    continue;
                }
                
                // If we have a person ID, insert the dose
                if ($personId) {
                    $stmt = $pdo->prepare("INSERT INTO doses (person_id, start_date, end_date, quantity, bodypart_label, value_mSv, unit_raw, employer, source, badge_serial, created_by)
                                           VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'import', ?, ?)
                                           ON DUPLICATE KEY UPDATE value_mSv = VALUES(value_mSv)");
                    $stmt->execute([$personId, $startDate, $endDate, $quantity, $bodypartLabel, $valueMsv, $data['Unit'], $doseEmployer, $ecn, $_SESSION['user_id']]);
                    
                    if ($stmt->rowCount() > 0) {
                        $stats['added']++;
                    }
                }
                
            } catch (Exception $e) {
                $stats['errors']++;
            }
        }
        
        $pdo->prepare("UPDATE imports SET committed_at = NOW() WHERE id = ?")->execute([$importId]);
        $pdo->commit();
        
        audit($pdo, 'import_commit', 'imports', $importId, $stats);
        unset($_SESSION['import_preview']);
        
        $message = "Import complete: {$stats['added']} doses added, {$stats['new_people']} new people created";
        if ($stats['external_employer_review'] > 0) {
            $message .= ", {$stats['external_employer_review']} need employer assignment";
        }
        if ($stats['merge_needed'] > 0) {
            $message .= ", {$stats['merge_needed']} need merge review";
        }
        if ($stats['errors'] > 0) {
            $message .= ", {$stats['errors']} errors";
        }
        
        redirect('?page=import', $message);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        redirect('?page=import', 'Import failed: ' . $e->getMessage(), 'error');
    }
}

// Main import page
else {
    $templates = $pdo->query("SELECT * FROM templates ORDER BY name")->fetchAll();
    
    echo "<h1>Import Dosimetry Data</h1>";
    
    echo "<div class='card'>";
    echo "<h2>Upload File</h2>";
    
    if (empty($templates)) {
        echo "<div class='alert alert-warning'>";
        echo "⚠ No import templates found. Please create a template in Settings first.";
        echo "</div>";
    } else {
        echo "<form method='post' action='?page=import&action=upload' enctype='multipart/form-data'>";
        
        echo "<div class='form-group'>";
        echo "<label>Select Template</label>";
        echo "<select name='template_id' required>";
        foreach ($templates as $t) {
            echo "<option value='{$t['id']}'>" . esc($t['name']) . " (" . esc($t['provider']) . ")</option>";
        }
        echo "</select>";
        echo "</div>";
        
        echo "<div class='form-group'>";
        echo "<label>Upload File (CSV/TSV)</label>";
        echo "<input type='file' name='file' accept='.csv,.tsv' required>";
        echo "</div>";
        
        echo "<button type='submit' class='btn btn-primary'>Preview Import</button>";
        echo "</form>";
    }
    
    echo "</div>";
    
    // Recent imports
    $imports = $pdo->query("SELECT i.*, u.username FROM imports i 
                           LEFT JOIN users u ON i.created_by = u.id 
                           ORDER BY i.created_at DESC LIMIT 10")->fetchAll();
    
    if (!empty($imports)) {
        echo "<div class='card'>";
        echo "<h2>Recent Imports</h2>";
        echo "<table>";
        echo "<thead><tr>";
        echo "<th>Filename</th><th>Provider</th><th>Rows</th><th>Status</th><th>Imported By</th><th>Date</th>";
        echo "</tr></thead>";
        echo "<tbody>";
        
        foreach ($imports as $imp) {
            echo "<tr>";
            echo "<td>" . esc($imp['filename']) . "</td>";
            echo "<td>" . esc($imp['provider']) . "</td>";
            echo "<td>{$imp['row_count']}</td>";
            echo "<td><span class='badge badge-success'>" . esc(ucfirst($imp['status'])) . "</span></td>";
            echo "<td>" . esc($imp['username']) . "</td>";
            echo "<td>" . date('d/m/Y H:i', strtotime($imp['created_at'])) . "</td>";
            echo "</tr>";
        }
        
        echo "</tbody></table>";
        echo "</div>";
    }
}
