<?php
/**
 * PDMS Module System
 * This file handles all module functionality
 * 
 * SAVE THIS AS: modules.php (same directory as index.php)
 */

// Get active modules for navigation
function getActiveModules($pdo) {
    $stmt = $pdo->query("SELECT * FROM modules WHERE is_active = 1 ORDER BY name");
    return $stmt->fetchAll();
}

// Check if current page is a module and handle it
function checkModulePage($pdo, $page, $settings) {
    $stmt = $pdo->prepare("SELECT * FROM modules WHERE slug = ? AND is_active = 1");
    $stmt->execute([$page]);
    $module = $stmt->fetch();
    
    if (!$module) return false;
    
    // Check permissions
    if ($module['required_role'] === 'admin' && $_SESSION['role'] !== 'admin') {
        redirect('?page=dashboard', 'Access denied', 'error');
    }
    if ($module['required_role'] === 'editor' && !in_array($_SESSION['role'], ['admin', 'editor'])) {
        redirect('?page=dashboard', 'Access denied', 'error');
    }
    
    // Execute module code
    ob_start();
    try {
        eval('?>' . $module['module_code']);
    } catch (Throwable $e) {
        echo "<div class='card' style='border-left:3px solid var(--error)'>";
        echo "<h2>Module Error: " . esc($module['name']) . "</h2>";
        echo "<p style='color:var(--error)'>Error: " . esc($e->getMessage()) . "</p>";
        echo "<p style='color:var(--text-secondary);margin-top:10px'>Line: " . $e->getLine() . "</p>";
        echo "<p style='color:var(--text-secondary)'>Edit this module in Settings â†’ Modules to fix the error.</p>";
        echo "</div>";
    }
    $content = ob_get_clean();
    
    // Render with full layout
    $activeModules = getActiveModules($pdo);
    $theme = $settings['theme'];
    ?>
    <!DOCTYPE html>
    <html lang="en" data-theme="<?= $theme ?>">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title><?= esc($settings['site_title']) ?></title>
        <style>
            :root {--bg-primary:#1a1a2e;--bg-secondary:#16213e;--bg-card:#0f3460;--text-primary:#eee;--text-secondary:#aaa;--accent:#e94560;--accent-hover:#d63447;--border:#2a2a3e;--success:#2ecc71;--warning:#f39c12;--error:#e74c3c}
            [data-theme="light"]{--bg-primary:#f5f5f5;--bg-secondary:#fff;--bg-card:#fff;--text-primary:#333;--text-secondary:#666;--border:#ddd}
            *{margin:0;padding:0;box-sizing:border-box}
            body{font-family:-apple-system,BlinkMacSystemFont,'Segoe UI',Arial,sans-serif;background:var(--bg-primary);color:var(--text-primary);line-height:1.6}
            .container{max-width:1400px;margin:0 auto;padding:20px}
            .header{background:var(--bg-secondary);border-bottom:1px solid var(--border);padding:15px 0;margin-bottom:30px}
            .header-content{max-width:1400px;margin:0 auto;padding:0 20px;display:flex;justify-content:space-between;align-items:center;flex-wrap:wrap}
            .logo{font-size:24px;font-weight:700;color:var(--accent)}
            .nav{display:flex;gap:20px;align-items:center;flex-wrap:wrap}
            .nav a{color:var(--text-primary);text-decoration:none;padding:8px 16px;border-radius:5px;transition:background 0.3s}
            .nav a:hover{background:var(--bg-card)}
            .btn{padding:10px 20px;border:none;border-radius:5px;cursor:pointer;font-size:14px;transition:all 0.3s;text-decoration:none;display:inline-block}
            .btn-primary{background:var(--accent);color:white}
            .btn-primary:hover{background:var(--accent-hover)}
            .btn-secondary{background:var(--bg-card);color:var(--text-primary)}
            .card{background:var(--bg-secondary);border:1px solid var(--border);border-radius:8px;padding:20px;margin-bottom:20px}
            .alert{padding:12px 20px;border-radius:5px;margin-bottom:20px}
            .alert-success{background:var(--success);color:white}
            .alert-error{background:var(--error);color:white}
            h1{margin-bottom:20px}h2{margin-bottom:15px}
            table{width:100%;border-collapse:collapse}
            th,td{padding:12px;text-align:left;border-bottom:1px solid var(--border)}
            th{background:var(--bg-card);font-weight:600}
            input,select,textarea{width:100%;padding:10px;border:1px solid var(--border);border-radius:5px;background:var(--bg-card);color:var(--text-primary);font-size:14px}
            .form-group{margin-bottom:15px}
            label{display:block;margin-bottom:5px;font-weight:500}
        </style>
    </head>
    <body>
        <div class="header">
            <div class="header-content">
                <div class="logo"><?= esc($settings['site_title']) ?></div>
                <div class="nav">
                    <a href="?page=dashboard">Dashboard</a>
                    <a href="?page=people">People</a>
                    <?php if (in_array($_SESSION['role'], ['admin', 'editor'])): ?>
                    <a href="?page=import">Import</a>
                    <a href="?page=merges">Merges</a>
                    <?php endif; ?>
                    <a href="?page=reports">Reports</a>
                    <?php foreach ($activeModules as $mod): 
                        if ($mod['menu_title']): ?>
                            <a href="?page=<?= esc($mod['slug']) ?>"><?= esc($mod['menu_icon']) ?> <?= esc($mod['menu_title']) ?></a>
                        <?php endif;
                    endforeach; ?>
                    <?php if ($_SESSION['role'] === 'admin'): ?>
                    <a href="?page=settings">Settings</a>
                    <a href="?page=audit">Audit</a>
                    <?php endif; ?>
                    <span style="color: var(--text-secondary);"><?= isset($_SESSION['username']) ? esc($_SESSION['username']) : 'User' ?></span>
                    <a href="?page=logout">Logout</a>
                </div>
            </div>
        </div>
        <div class="container">
            <?= flashMessage() ?>
            <?= $content ?>
        </div>
    </body>
    </html>
    <?php
    return true; // Module was handled
}

// Handle module settings (add/edit/delete)
function handleModuleSettings($pdo, $action, $settings, $activeModules) {
    $tab = $_GET['tab'] ?? 'general';
    
    // Handle POST actions
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'add_module') {
        $slug = preg_replace('/[^a-z0-9_]/', '', strtolower(str_replace(' ', '_', $_POST['name'])));
        $installSql = trim($_POST['install_sql'] ?? '');
        
        if ($installSql) {
            try { $pdo->exec($installSql); }
            catch (Exception $e) { redirect('?page=settings&tab=modules', 'SQL Error: ' . $e->getMessage(), 'error'); }
        }
        
        $pdo->prepare("INSERT INTO modules (name, slug, description, version, module_code, menu_title, menu_icon, required_role, install_sql, created_by) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)")->execute([
            $_POST['name'], $slug, $_POST['description'], $_POST['version'], $_POST['module_code'], 
            $_POST['menu_title'], $_POST['menu_icon'], $_POST['required_role'], $installSql, $_SESSION['user_id']
        ]);
        audit($pdo, 'add_module', 'modules', $pdo->lastInsertId(), ['name' => $_POST['name']]);
        redirect('?page=settings&tab=modules', 'Module created successfully!');
    }
    
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action === 'edit_module') {
        $pdo->prepare("UPDATE modules SET name=?, description=?, version=?, module_code=?, menu_title=?, menu_icon=?, required_role=? WHERE id=?")->execute([
            $_POST['name'], $_POST['description'], $_POST['version'], $_POST['module_code'], 
            $_POST['menu_title'], $_POST['menu_icon'], $_POST['required_role'], $_POST['module_id']
        ]);
        audit($pdo, 'edit_module', 'modules', $_POST['module_id'], ['name' => $_POST['name']]);
        redirect('?page=settings&tab=modules', 'Module updated successfully!');
    }
    
    if ($action === 'toggle_module') {
        $pdo->prepare("UPDATE modules SET is_active = NOT is_active WHERE id = ?")->execute([$_GET['id']]);
        redirect('?page=settings&tab=modules', 'Module status updated');
    }
    
    if ($action === 'delete_module') {
        $pdo->prepare("DELETE FROM modules WHERE id = ?")->execute([$_GET['id']]);
        audit($pdo, 'delete_module', 'modules', $_GET['id'], []);
        redirect('?page=settings&tab=modules', 'Module deleted');
    }
    
    // Render module settings tab
    if ($tab === 'modules') {
        $modules = $pdo->query("SELECT m.*, u.username as creator FROM modules m LEFT JOIN users u ON m.created_by = u.id ORDER BY m.name")->fetchAll();
        
        if ($action === 'add') {
            echo "<div class='card'><h2>Add Module</h2>";
            echo "<p style='color:var(--text-secondary);margin-bottom:20px'>Create a custom module to extend PDMS functionality.</p>";
            echo "<form method='post' action='?page=settings&action=add_module'>";
            echo "<div style='display:grid;grid-template-columns:1fr 1fr;gap:15px'>";
            echo "<div class='form-group'><label>Module Name *</label><input type='text' name='name' required placeholder='e.g., Custom Report'></div>";
            echo "<div class='form-group'><label>Version</label><input type='text' name='version' value='1.0' placeholder='1.0'></div>";
            echo "</div>";
            echo "<div class='form-group'><label>Description</label><textarea name='description' rows='2' placeholder='What does this module do?'></textarea></div>";
            echo "<div style='display:grid;grid-template-columns:1fr 1fr 1fr;gap:15px'>";
            echo "<div class='form-group'><label>Menu Title (optional)</label><input type='text' name='menu_title' placeholder='Appears in navigation'></div>";
            echo "<div class='form-group'><label>Menu Icon</label><input type='text' name='menu_icon' placeholder='ðŸ“Š'></div>";
            echo "<div class='form-group'><label>Required Role</label><select name='required_role'>";
            echo "<option value='viewer'>Viewer (all users)</option>";
            echo "<option value='editor'>Editor & Admin</option>";
            echo "<option value='admin'>Admin Only</option>";
            echo "</select></div></div>";
            echo "<div class='form-group'><label>Installation SQL (optional)</label><textarea name='install_sql' rows='3' placeholder='CREATE TABLE IF NOT EXISTS ...'></textarea>";
            echo "<small style='color:var(--text-secondary)'>Database changes to run when module is created</small></div>";
            echo "<div class='form-group'><label>Module Code (PHP) *</label>";
            echo "<textarea name='module_code' rows='18' required style='font-family:monospace;font-size:13px'></textarea>";
            echo "<small style='color:var(--text-secondary)'><strong>Available:</strong> \$pdo (database), \$settings (site settings), \$_SESSION (user session), and all helper functions (esc(), redirect(), audit(), etc.)</small></div>";
            echo "<button type='submit' class='btn btn-primary'>Create Module</button> ";
            echo "<a href='?page=settings&tab=modules' class='btn btn-secondary'>Cancel</a>";
            echo "</form></div>";
            
            // Example code
            echo "<div class='card' style='background:var(--bg-card)'><h3>Example Module Code</h3>";
            echo "<pre style='background:var(--bg-primary);padding:15px;border-radius:5px;overflow-x:auto;font-size:12px'>";
            echo htmlspecialchars('<h1>My Custom Module</h1>

<div class="card">
    <h2>Database Query Example</h2>
    <?php
    $count = $pdo->query("SELECT COUNT(*) FROM people WHERE status = \'active\'")->fetchColumn();
    echo "<p>Active People: <strong>$count</strong></p>";
    echo "<p>Current User: <strong>" . esc($_SESSION["username"]) . "</strong></p>";
    echo "<p>User Role: <strong>" . esc($_SESSION["role"]) . "</strong></p>";
    ?>
</div>

<div class="card">
    <h2>Form Example</h2>
    <?php
    if ($_SERVER["REQUEST_METHOD"] === "POST") {
        $value = $_POST["my_field"];
        // Process data, save to database, etc.
        redirect("?page=my_custom_module", "Data saved: $value");
    }
    ?>
    <form method="post">
        <div class="form-group">
            <label>Input Field</label>
            <input type="text" name="my_field" required>
        </div>
        <button type="submit" class="btn btn-primary">Submit</button>
    </form>
</div>

<div class="card">
    <h2>Table Example</h2>
    <?php
    $people = $pdo->query("SELECT * FROM people LIMIT 10")->fetchAll();
    echo "<table><thead><tr><th>Name</th><th>Department</th><th>Status</th></tr></thead><tbody>";
    foreach ($people as $p) {
        echo "<tr>";
        echo "<td>" . esc($p["forename"] . " " . $p["surname"]) . "</td>";
        echo "<td>" . esc($p["department_id"]) . "</td>";
        echo "<td>" . esc($p["status"]) . "</td>";
        echo "</tr>";
    }
    echo "</tbody></table>";
    ?>
</div>');
            echo "</pre></div>";
            
        } elseif ($action === 'edit') {
            $stmt = $pdo->prepare("SELECT * FROM modules WHERE id = ?");
            $stmt->execute([$_GET['id']]);
            $mod = $stmt->fetch();
            
            echo "<div class='card'><h2>Edit Module</h2>";
            echo "<form method='post' action='?page=settings&action=edit_module'>";
            echo "<input type='hidden' name='module_id' value='{$mod['id']}'>";
            echo "<div style='display:grid;grid-template-columns:1fr 1fr;gap:15px'>";
            echo "<div class='form-group'><label>Module Name *</label><input type='text' name='name' value='" . esc($mod['name']) . "' required></div>";
            echo "<div class='form-group'><label>Version</label><input type='text' name='version' value='" . esc($mod['version']) . "'></div>";
            echo "</div>";
            echo "<div class='form-group'><label>Description</label><textarea name='description' rows='2'>" . esc($mod['description']) . "</textarea></div>";
            echo "<div style='display:grid;grid-template-columns:1fr 1fr 1fr;gap:15px'>";
            echo "<div class='form-group'><label>Menu Title</label><input type='text' name='menu_title' value='" . esc($mod['menu_title']) . "'></div>";
            echo "<div class='form-group'><label>Menu Icon</label><input type='text' name='menu_icon' value='" . esc($mod['menu_icon']) . "'></div>";
            echo "<div class='form-group'><label>Required Role</label><select name='required_role'>";
            echo "<option value='viewer'" . ($mod['required_role']=='viewer'?' selected':'') . ">Viewer (all users)</option>";
            echo "<option value='editor'" . ($mod['required_role']=='editor'?' selected':'') . ">Editor & Admin</option>";
            echo "<option value='admin'" . ($mod['required_role']=='admin'?' selected':'') . ">Admin Only</option>";
            echo "</select></div></div>";
            echo "<div class='form-group'><label>Module Code (PHP) *</label>";
            echo "<textarea name='module_code' rows='18' required style='font-family:monospace;font-size:13px'>" . esc($mod['module_code']) . "</textarea></div>";
            echo "<button type='submit' class='btn btn-primary'>Update Module</button> ";
            echo "<a href='?page=settings&tab=modules' class='btn btn-secondary'>Cancel</a>";
            echo "</form></div>";
            
        } else {
            // List all modules
            echo "<div class='card'><h2>Module Manager</h2>";
            echo "<p style='margin-bottom:20px;color:var(--text-secondary)'>Modules extend PDMS with custom pages and functionality without modifying core code.</p>";
            echo "<a href='?page=settings&tab=modules&action=add' class='btn btn-primary' style='margin-bottom:20px'>Add New Module</a>";
            
            if (empty($modules)) {
                echo "<p>No modules created yet. Create your first module to extend PDMS functionality!</p>";
            } else {
                echo "<table><thead><tr><th>Name</th><th>Version</th><th>Menu</th><th>Role</th><th>Status</th><th>Created</th><th>Actions</th></tr></thead><tbody>";
                foreach ($modules as $m) {
                    echo "<tr>";
                    echo "<td><strong>" . esc($m['name']) . "</strong><br><small style='color:var(--text-secondary)'>URL: ?page=" . esc($m['slug']) . "</small></td>";
                    echo "<td>" . esc($m['version']) . "</td>";
                    echo "<td>" . ($m['menu_title'] ? esc($m['menu_icon'] . ' ' . $m['menu_title']) : '<span style="color:var(--text-secondary)">Hidden</span>') . "</td>";
                    echo "<td><span style='padding:3px 8px;background:var(--bg-card);border-radius:3px;font-size:11px;text-transform:capitalize'>" . esc($m['required_role']) . "</span></td>";
                    echo "<td><span style='color:" . ($m['is_active'] ? 'var(--success)' : 'var(--error)') . ";font-weight:600'>" . ($m['is_active'] ? 'âœ“ Active' : 'âœ— Inactive') . "</span></td>";
                    echo "<td><small>" . esc($m['creator']) . "<br>" . date('d/m/Y', strtotime($m['created_at'])) . "</small></td>";
                    echo "<td style='white-space:nowrap'>";
                    if ($m['is_active']) {
                        echo "<a href='?page=" . esc($m['slug']) . "' class='btn btn-secondary' style='padding:5px 10px;font-size:12px' target='_blank'>View</a> ";
                    }
                    echo "<a href='?page=settings&tab=modules&action=edit&id=" . $m['id'] . "' class='btn btn-secondary' style='padding:5px 10px;font-size:12px'>Edit</a> ";
                    echo "<a href='?page=settings&tab=modules&action=toggle_module&id=" . $m['id'] . "' class='btn btn-secondary' style='padding:5px 10px;font-size:12px'>" . ($m['is_active'] ? 'Disable' : 'Enable') . "</a> ";
                    echo "<a href='?page=settings&tab=modules&action=delete_module&id=" . $m['id'] . "' class='btn btn-secondary' style='padding:5px 10px;font-size:12px;background:var(--error)' onclick='return confirm(\"Permanently delete this module? This cannot be undone.\")'>Delete</a>";
                    echo "</td></tr>";
                }
                echo "</tbody></table>";
            }
            echo "</div>";
        }
        
        return true; // Tab was handled
    }
    
    return false; // Not modules tab, continue with other settings
}

// Add module tab to settings navigation
function addModuleTabToSettings($currentTab) {
    echo "<a href='?page=settings&tab=modules' style='padding:10px 0;border-bottom:3px solid " . ($currentTab === 'modules' ? 'var(--accent)' : 'transparent') . ";text-decoration:none;color:var(--text-primary);display:inline-block;margin-right:20px'>Modules</a>";
}
// Allow modules to register hooks
function register_module_hooks($pdo) {
    $modules = $pdo->query("SELECT * FROM modules WHERE is_active = 1")->fetchAll();
    foreach ($modules as $module) {
        // Check if module has hook registration
        if (stripos($module['module_code'], 'add_module_hook') !== false) {
            try {
                eval('?>' . $module['module_code']);
            } catch (Exception $e) {
                // Silently fail
            }
        }
    }
}