<?php
/**
 * PDMS Autoinstaller
 * Complete setup wizard for Personal Dosimetry Management System
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();

// Check if already installed
if (file_exists(__DIR__ . '/config.php') && !isset($_GET['force'])) {
    die('PDMS is already installed. Delete config.php to reinstall, or add ?force=1 to reinstall anyway.');
}

$step = $_GET['step'] ?? 1;
$errors = [];
$success = [];

// Step 1: Database connection test
if ($step == 1 && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $db_host = $_POST['db_host'];
    $db_name = $_POST['db_name'];
    $db_user = $_POST['db_user'];
    $db_pass = $_POST['db_pass'];
    $site_url = rtrim($_POST['site_url'], '/');
    
    try {
        $pdo = new PDO(
            "mysql:host=$db_host;charset=utf8mb4",
            $db_user,
            $db_pass,
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
        
        // Create database if it doesn't exist
        $pdo->exec("CREATE DATABASE IF NOT EXISTS `$db_name` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        $pdo->exec("USE `$db_name`");
        
        // Store in session for next step
        $_SESSION['install'] = [
            'db_host' => $db_host,
            'db_name' => $db_name,
            'db_user' => $db_user,
            'db_pass' => $db_pass,
            'site_url' => $site_url
        ];
        
        header('Location: install.php?step=2');
        exit;
        
    } catch (PDOException $e) {
        $errors[] = "Database connection failed: " . $e->getMessage();
    }
}

// Step 2: Create tables and initial data
if ($step == 2 && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $install = $_SESSION['install'];
    
    try {
        $pdo = new PDO(
            "mysql:host={$install['db_host']};dbname={$install['db_name']};charset=utf8mb4",
            $install['db_user'],
            $install['db_pass'],
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
        
        // Create all tables
        $sql = "
        -- Users table
        CREATE TABLE IF NOT EXISTS users (
            id INT PRIMARY KEY AUTO_INCREMENT,
            username VARCHAR(100) UNIQUE NOT NULL,
            password VARCHAR(255) NOT NULL,
            role VARCHAR(20) NOT NULL,
            is_active TINYINT(1) DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Departments table
        CREATE TABLE IF NOT EXISTS departments (
            id INT PRIMARY KEY AUTO_INCREMENT,
            name VARCHAR(255) NOT NULL,
            ecn VARCHAR(100),
            is_active TINYINT(1) DEFAULT 1,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Employer contacts table
        CREATE TABLE IF NOT EXISTS employer_contacts (
            id INT PRIMARY KEY AUTO_INCREMENT,
            employer_name VARCHAR(255) NOT NULL UNIQUE,
            contact_name VARCHAR(255),
            rps_name VARCHAR(255),
            contact_email VARCHAR(255),
            contact_phone VARCHAR(100),
            address TEXT,
            is_active TINYINT(1) DEFAULT 1,
            notes TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- People table
        CREATE TABLE IF NOT EXISTS people (
            id INT PRIMARY KEY AUTO_INCREMENT,
            department_id INT NOT NULL,
            forename VARCHAR(100) NOT NULL,
            surname VARCHAR(100) NOT NULL,
            ecn VARCHAR(100),
            date_of_birth DATE,
            national_insurance VARCHAR(20),
            emergency_contact VARCHAR(255),
            emergency_phone VARCHAR(100),
            employer_2_id INT,
            employer_3_id INT,
            last_dose_employer_1 DATE,
            last_dose_employer_2 DATE,
            last_dose_employer_3 DATE,
            status VARCHAR(20) DEFAULT 'active',
            started_at DATE,
            left_at DATE,
            terminated_by INT,
            terminated_at DATETIME,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (department_id) REFERENCES departments(id),
            FOREIGN KEY (employer_2_id) REFERENCES employer_contacts(id) ON DELETE SET NULL,
            FOREIGN KEY (employer_3_id) REFERENCES employer_contacts(id) ON DELETE SET NULL,
            FOREIGN KEY (terminated_by) REFERENCES users(id),
            INDEX idx_name (surname, forename),
            INDEX idx_status (status),
            INDEX idx_department (department_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Doses table
        CREATE TABLE IF NOT EXISTS doses (
            id INT PRIMARY KEY AUTO_INCREMENT,
            person_id INT NOT NULL,
            start_date DATE NOT NULL,
            end_date DATE NOT NULL,
            quantity VARCHAR(20) NOT NULL,
            bodypart_label VARCHAR(100) NOT NULL,
            value_mSv DECIMAL(10,4) NOT NULL,
            unit_raw VARCHAR(20),
            employer VARCHAR(255),
            source VARCHAR(20) DEFAULT 'manual',
            badge_serial VARCHAR(100),
            created_by INT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (person_id) REFERENCES people(id) ON DELETE CASCADE,
            FOREIGN KEY (created_by) REFERENCES users(id),
            UNIQUE KEY unique_dose (person_id, start_date, end_date, quantity, bodypart_label),
            INDEX idx_person_date (person_id, start_date),
            INDEX idx_date_range (start_date, end_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Notes table
        CREATE TABLE IF NOT EXISTS notes (
            id INT PRIMARY KEY AUTO_INCREMENT,
            person_id INT NOT NULL,
            author_id INT NOT NULL,
            text TEXT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (person_id) REFERENCES people(id) ON DELETE CASCADE,
            FOREIGN KEY (author_id) REFERENCES users(id),
            INDEX idx_person (person_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Templates table
        CREATE TABLE IF NOT EXISTS templates (
            id INT PRIMARY KEY AUTO_INCREMENT,
            name VARCHAR(255) NOT NULL,
            provider VARCHAR(100) NOT NULL,
            titles_in_first_row TINYINT(1) DEFAULT 1,
            column_map TEXT NOT NULL,
            unit_map TEXT NOT NULL,
            bodypart_map TEXT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Imports table
        CREATE TABLE IF NOT EXISTS imports (
            id INT PRIMARY KEY AUTO_INCREMENT,
            provider VARCHAR(100),
            filename VARCHAR(255),
            row_count INT DEFAULT 0,
            status VARCHAR(50) DEFAULT 'pending',
            created_by INT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            committed_at DATETIME,
            FOREIGN KEY (created_by) REFERENCES users(id),
            INDEX idx_status (status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Import rows table (for merge queue)
        CREATE TABLE IF NOT EXISTS import_rows (
            id INT PRIMARY KEY AUTO_INCREMENT,
            import_id INT NOT NULL,
            raw_json TEXT NOT NULL,
            status VARCHAR(50) DEFAULT 'pending',
            message TEXT,
            person_match_id INT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (import_id) REFERENCES imports(id) ON DELETE CASCADE,
            FOREIGN KEY (person_match_id) REFERENCES people(id) ON DELETE SET NULL,
            INDEX idx_status (status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Merge rules table
        CREATE TABLE IF NOT EXISTS merge_rules (
            id INT PRIMARY KEY AUTO_INCREMENT,
            key_hash VARCHAR(64) NOT NULL UNIQUE,
            person_id INT NOT NULL,
            accepted_by INT NOT NULL,
            accepted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (person_id) REFERENCES people(id) ON DELETE CASCADE,
            FOREIGN KEY (accepted_by) REFERENCES users(id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Triggers table
        CREATE TABLE IF NOT EXISTS triggers (
            id INT PRIMARY KEY AUTO_INCREMENT,
            department_id INT NOT NULL,
            quantity VARCHAR(20) NOT NULL,
            monthly_mSv DECIMAL(10,4) NOT NULL,
            ytd_mSv DECIMAL(10,4) NOT NULL,
            UNIQUE KEY unique_trigger (department_id, quantity),
            FOREIGN KEY (department_id) REFERENCES departments(id) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Dose sharing log table
        CREATE TABLE IF NOT EXISTS dose_sharing_log (
            id INT PRIMARY KEY AUTO_INCREMENT,
            person_id INT NOT NULL,
            employer_name VARCHAR(255),
            action_type VARCHAR(50),
            sent_by INT,
            notes TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (person_id) REFERENCES people(id) ON DELETE CASCADE,
            FOREIGN KEY (sent_by) REFERENCES users(id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Audit log table
        CREATE TABLE IF NOT EXISTS audit (
            id INT PRIMARY KEY AUTO_INCREMENT,
            user_id INT,
            action VARCHAR(100) NOT NULL,
            entity VARCHAR(50) NOT NULL,
            entity_id INT,
            payload TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
            INDEX idx_entity (entity, entity_id),
            INDEX idx_created (created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Modules table
        CREATE TABLE IF NOT EXISTS modules (
            id INT PRIMARY KEY AUTO_INCREMENT,
            name VARCHAR(255) NOT NULL,
            slug VARCHAR(100) NOT NULL UNIQUE,
            description TEXT,
            version VARCHAR(20) DEFAULT '1.0',
            module_code LONGTEXT NOT NULL,
            is_active TINYINT(1) DEFAULT 1,
            menu_title VARCHAR(100),
            menu_icon VARCHAR(50),
            required_role VARCHAR(20) DEFAULT 'viewer',
            install_sql TEXT,
            created_by INT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            FOREIGN KEY (created_by) REFERENCES users(id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        
        -- Settings table
        CREATE TABLE IF NOT EXISTS settings (
            id INT PRIMARY KEY AUTO_INCREMENT,
            site_title VARCHAR(255) DEFAULT 'PDMS',
            footer_text VARCHAR(255),
            theme VARCHAR(20) DEFAULT 'dark',
            employers TEXT,
            ecn_list TEXT,
            bodypart_config TEXT,
            enable_nonreturn_flagging TINYINT(1) DEFAULT 1
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
        ";
        
        // Execute table creation
        $pdo->exec($sql);
        
        // Insert default settings
        $defaultBodyparts = json_encode([
            'Hp10' => ['Body', 'Chest', 'Abdomen'],
            'Hp007' => ['Skin'],
            'Hp3' => ['Eye', 'Finger Ring Left', 'Finger Ring Right', 'Wrist']
        ]);
        
        $pdo->exec("INSERT INTO settings (site_title, footer_text, theme, employers, ecn_list, bodypart_config, enable_nonreturn_flagging) 
                    VALUES ('PDMS', 'Personal Dosimetry Management System', 'dark', '[]', '[]', '$defaultBodyparts', 1)");
        
        // Create admin user
        $admin_username = $_POST['admin_username'];
        $admin_password = password_hash($_POST['admin_password'], PASSWORD_DEFAULT);
        $pdo->prepare("INSERT INTO users (username, password, role, is_active) VALUES (?, ?, 'admin', 1)")
             ->execute([$admin_username, $admin_password]);
        
        // Create default department
        $pdo->exec("INSERT INTO departments (name, ecn, is_active) VALUES ('General', 'GEN001', 1)");
        
        // Create default import template
        $columnMap = json_encode([
            'Forename' => 'Forename',
            'Surname' => 'Surname',
            'ECN' => 'ECN',
            'Department' => 'Department',
            'PeriodStart' => 'Period Start',
            'PeriodEnd' => 'Period End',
            'Value' => 'Dose Value',
            'Unit' => 'Unit',
            'BadgeType' => 'Badge Type',
            'Employer' => 'Employer'
        ]);
        
        $unitMap = json_encode([
            'uSv' => ['op' => 'divide', 'by' => 1000],
            'μSv' => ['op' => 'divide', 'by' => 1000],
            'mSv' => ['op' => 'multiply', 'by' => 1]
        ]);
        
        $bodypartMap = json_encode([
            'BODY' => 'Hp10',
            'CHEST' => 'Hp10',
            'SKIN' => 'Hp007',
            'EYE' => 'Hp3',
            'FINGER' => 'Hp3',
            'WRIST' => 'Hp3'
        ]);
        
        $pdo->prepare("INSERT INTO templates (name, provider, titles_in_first_row, column_map, unit_map, bodypart_map) 
                       VALUES (?, ?, 1, ?, ?, ?)")
             ->execute(['Standard Template', 'Generic', $columnMap, $unitMap, $bodypartMap]);
        
        // Store admin username for next step
        $_SESSION['install']['admin_username'] = $admin_username;
        
        header('Location: install.php?step=3');
        exit;
        
    } catch (PDOException $e) {
        $errors[] = "Database setup failed: " . $e->getMessage();
    }
}

// Step 3: Create config file
if ($step == 3 && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $install = $_SESSION['install'];
    
    $config_content = "<?php
/**
 * PDMS Configuration File
 * Generated by installer on " . date('Y-m-d H:i:s') . "
 */

// Database Configuration
define('DB_HOST', '{$install['db_host']}');
define('DB_NAME', '{$install['db_name']}');
define('DB_USER', '{$install['db_user']}');
define('DB_PASS', '{$install['db_pass']}');

// Site Configuration
define('SITE_URL', '{$install['site_url']}');

// Session Configuration
ini_set('session.cookie_httponly', 1);
ini_set('session.use_strict_mode', 1);
ini_set('session.cookie_samesite', 'Strict');
";
    
    if (file_put_contents(__DIR__ . '/config.php', $config_content)) {
        $success[] = "Configuration file created successfully!";
        $_SESSION['install']['complete'] = true;
    } else {
        $errors[] = "Failed to create config.php. Please check file permissions.";
    }
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PDMS Installer</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .installer {
            background: white;
            border-radius: 10px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            padding: 40px;
            width: 100%;
            max-width: 600px;
        }
        h1 {
            color: #333;
            margin-bottom: 10px;
            font-size: 28px;
        }
        .step-indicator {
            color: #666;
            margin-bottom: 30px;
            font-size: 14px;
        }
        .progress-bar {
            height: 4px;
            background: #e0e0e0;
            border-radius: 2px;
            margin-bottom: 30px;
            overflow: hidden;
        }
        .progress-fill {
            height: 100%;
            background: #667eea;
            transition: width 0.3s;
        }
        .form-group {
            margin-bottom: 20px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            color: #333;
            font-weight: 500;
        }
        input, select {
            width: 100%;
            padding: 12px;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 14px;
        }
        input:focus {
            outline: none;
            border-color: #667eea;
        }
        button {
            width: 100%;
            padding: 14px;
            background: #667eea;
            color: white;
            border: none;
            border-radius: 5px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: background 0.3s;
        }
        button:hover {
            background: #5568d3;
        }
        .error {
            background: #fee;
            color: #c33;
            padding: 12px;
            border-radius: 5px;
            margin-bottom: 20px;
            border-left: 3px solid #c33;
        }
        .success {
            background: #efe;
            color: #363;
            padding: 12px;
            border-radius: 5px;
            margin-bottom: 20px;
            border-left: 3px solid #363;
        }
        .info {
            background: #e7f3ff;
            color: #014361;
            padding: 12px;
            border-radius: 5px;
            margin-bottom: 20px;
            border-left: 3px solid #014361;
        }
        .complete-box {
            text-align: center;
            padding: 40px 20px;
        }
        .complete-box h2 {
            color: #2ecc71;
            margin-bottom: 20px;
            font-size: 32px;
        }
        .complete-box p {
            color: #666;
            margin-bottom: 30px;
            line-height: 1.6;
        }
        .btn-primary {
            background: #2ecc71;
            text-decoration: none;
            display: inline-block;
            padding: 14px 40px;
            border-radius: 5px;
            color: white;
            font-weight: 600;
        }
        .btn-primary:hover {
            background: #27ae60;
        }
    </style>
</head>
<body>
    <div class="installer">
        <?php if (!empty($errors)): ?>
            <?php foreach ($errors as $error): ?>
                <div class="error"><?= htmlspecialchars($error) ?></div>
            <?php endforeach; ?>
        <?php endif; ?>
        
        <?php if (!empty($success)): ?>
            <?php foreach ($success as $msg): ?>
                <div class="success"><?= htmlspecialchars($msg) ?></div>
            <?php endforeach; ?>
        <?php endif; ?>
        
        <?php if ($step == 1): ?>
            <h1>PDMS Installation</h1>
            <div class="step-indicator">Step 1 of 3: Database Configuration</div>
            <div class="progress-bar"><div class="progress-fill" style="width: 33%"></div></div>
            
            <div class="info">
                <strong>Welcome!</strong> This wizard will help you set up your Personal Dosimetry Management System.
            </div>
            
            <form method="post">
                <div class="form-group">
                    <label>Database Host</label>
                    <input type="text" name="db_host" value="localhost" required>
                </div>
                
                <div class="form-group">
                    <label>Database Name</label>
                    <input type="text" name="db_name" value="pdms" required>
                </div>
                
                <div class="form-group">
                    <label>Database Username</label>
                    <input type="text" name="db_user" required>
                </div>
                
                <div class="form-group">
                    <label>Database Password</label>
                    <input type="password" name="db_pass">
                </div>
                
                <div class="form-group">
                    <label>Site URL</label>
                    <input type="url" name="site_url" value="<?= htmlspecialchars('http://' . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF'])) ?>" required>
                </div>
                
                <button type="submit">Continue →</button>
            </form>
            
        <?php elseif ($step == 2): ?>
            <h1>PDMS Installation</h1>
            <div class="step-indicator">Step 2 of 3: Database Setup</div>
            <div class="progress-bar"><div class="progress-fill" style="width: 66%"></div></div>
            
            <div class="info">
                <strong>Database connected!</strong> Now let's create your admin account.
            </div>
            
            <form method="post">
                <div class="form-group">
                    <label>Admin Username</label>
                    <input type="text" name="admin_username" value="admin" required>
                </div>
                
                <div class="form-group">
                    <label>Admin Password</label>
                    <input type="password" name="admin_password" required minlength="6">
                </div>
                
                <button type="submit">Create Database & Admin Account →</button>
            </form>
            
        <?php elseif ($step == 3 && !isset($_SESSION['install']['complete'])): ?>
            <h1>PDMS Installation</h1>
            <div class="step-indicator">Step 3 of 3: Finalize Installation</div>
            <div class="progress-bar"><div class="progress-fill" style="width: 100%"></div></div>
            
            <div class="info">
                <strong>Almost done!</strong> Click below to create the configuration file.
            </div>
            
            <form method="post">
                <button type="submit">Complete Installation →</button>
            </form>
            
        <?php else: ?>
            <div class="complete-box">
                <h2>✓ Installation Complete!</h2>
                <p>
                    Your PDMS installation is complete and ready to use.<br>
                    <strong>Admin Username:</strong> <?= htmlspecialchars($_SESSION['install']['admin_username']) ?>
                </p>
                <a href="index.php" class="btn-primary">Go to PDMS →</a>
            </div>
            <?php unset($_SESSION['install']); ?>
        <?php endif; ?>
    </div>
</body>
</html>
