<?php
/**
 * Personal Dosimetry Management System (PDMS)
 * Complete Rebuild - Fully Working System
 * Version 2.0 - 2025
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Check installation
if (!file_exists(__DIR__ . '/config.php')) {
    die('Not installed. Please run <a href="install.php">install.php</a> first.');
}

require_once __DIR__ . '/config.php';

// Start session with security settings
session_start();
if (!isset($_SESSION['regenerated'])) {
    session_regenerate_id(true);
    $_SESSION['regenerated'] = true;
}

// Session timeout (30 minutes)
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity'] > 1800)) {
    session_destroy();
    header('Location: ' . SITE_URL);
    exit;
}
$_SESSION['last_activity'] = time();

// Database connection
try {
    $pdo = new PDO(
        'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4',
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC
        ]
    );
} catch (PDOException $e) {
    die('Database connection failed. Please check your configuration.');
}

// ==================== HELPER FUNCTIONS ====================

function esc($str) {
    return htmlspecialchars($str ?? '', ENT_QUOTES, 'UTF-8');
}

function redirect($url, $message = null, $type = 'success') {
    if ($message) {
        $_SESSION['flash_message'] = $message;
        $_SESSION['flash_type'] = $type;
    }
    header('Location: ' . SITE_URL . $url);
    exit;
}

function flashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $msg = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'] ?? 'success';
        unset($_SESSION['flash_message'], $_SESSION['flash_type']);
        return "<div class='alert alert-$type'>$msg</div>";
    }
    return '';
}

function requireLogin() {
    if (!isset($_SESSION['user_id'])) {
        redirect('?page=login');
    }
}

function requireRole($roles) {
    requireLogin();
    $roles = is_array($roles) ? $roles : [$roles];
    if (!in_array($_SESSION['role'], $roles)) {
        redirect('?page=dashboard', 'Access denied', 'error');
    }
}

function audit($pdo, $action, $entity, $entity_id, $payload = []) {
    $stmt = $pdo->prepare("INSERT INTO audit (user_id, action, entity, entity_id, payload) VALUES (?, ?, ?, ?, ?)");
    $stmt->execute([
        $_SESSION['user_id'] ?? null,
        $action,
        $entity,
        $entity_id,
        json_encode($payload)
    ]);
}

function parseDate($date) {
    $formats = ['Y-m-d', 'd/m/Y', 'm/d/Y', 'd-m-Y', 'Y/m/d'];
    foreach ($formats as $format) {
        $d = DateTime::createFromFormat($format, trim($date));
        if ($d && $d->format($format) === trim($date)) {
            return $d->format('Y-m-d');
        }
    }
    return null;
}

function normalizeUnit($value, $unit, $unitMap) {
    $value = floatval($value);
    $unit = strtoupper(trim($unit));
    
    if (isset($unitMap[$unit])) {
        $map = $unitMap[$unit];
        if ($map['op'] === 'divide') {
            $value /= $map['by'];
        } elseif ($map['op'] === 'multiply') {
            $value *= $map['by'];
        }
    }
    
    return $value;
}

function deriveQuantityFromBodypart($bodypart, $bodypartMap) {
    $bodypart = strtoupper(trim($bodypart));
    
    foreach ($bodypartMap as $key => $qty) {
        if (strtoupper($key) === $bodypart) {
            return [$qty, $key];
        }
    }
    
    // Default to Hp10/Body if not found
    return ['Hp10', 'Body'];
}

function generateMergeKey($forename, $surname, $dept, $ecn) {
    $key = trim(strtolower($forename)) . '|' . 
           trim(strtolower($surname)) . '|' . 
           trim(strtolower($dept)) . '|' . 
           trim($ecn);
    return hash('sha256', $key);
}

// ==================== LOAD MODULE SYSTEM ====================

if (file_exists(__DIR__ . '/modules.php')) {
    require_once __DIR__ . '/modules.php';
}

// ==================== ROUTING ====================

$page = $_GET['page'] ?? 'dashboard';
$action = $_GET['action'] ?? '';

// ==================== LOGIN PAGE ====================

if ($page === 'login') {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $username = trim($_POST['username'] ?? '');
        $password = $_POST['password'] ?? '';
        
        $stmt = $pdo->prepare("SELECT * FROM users WHERE username = ? AND is_active = 1");
        $stmt->execute([$username]);
        $user = $stmt->fetch();
        
        if ($user && password_verify($password, $user['password'])) {
            session_regenerate_id(true);
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['role'] = $user['role'];
            $_SESSION['regenerated'] = true;
            
            audit($pdo, 'login', 'users', $user['id'], ['username' => $username]);
            redirect('?page=dashboard', 'Welcome back, ' . $username . '!');
        } else {
            $error = 'Invalid credentials';
        }
    }
    
    if (isset($_SESSION['user_id'])) {
        redirect('?page=dashboard');
    }
    
    include __DIR__ . '/views/login.php';
    exit;
}

// ==================== LOGOUT ====================

if ($page === 'logout') {
    audit($pdo, 'logout', 'users', $_SESSION['user_id'] ?? 0, []);
    session_destroy();
    redirect('?page=login', 'Logged out successfully');
}

// ==================== REQUIRE LOGIN FOR ALL OTHER PAGES ====================

requireLogin();

// ==================== LOAD SETTINGS ====================

$stmt = $pdo->query("SELECT * FROM settings LIMIT 1");
$settings = $stmt->fetch();

if (!$settings) {
    die('Settings not found. Please reinstall the system.');
}

// Decode JSON fields
$settings['employers'] = json_decode($settings['employers'] ?? '[]', true) ?: [];
$settings['ecn_list'] = json_decode($settings['ecn_list'] ?? '[]', true) ?: [];
$settings['bodypart_config'] = json_decode($settings['bodypart_config'] ?? '{"Hp10":["Body"],"Hp007":["Skin"],"Hp3":["Eye"]}', true) ?: ['Hp10' => ['Body']];

// ==================== CHECK FOR MODULE PAGE ====================

if (function_exists('checkModulePage')) {
    $activeModules = getActiveModules($pdo);
    if (checkModulePage($pdo, $page, $settings)) {
        exit; // Module handled the request
    }
} else {
    $activeModules = [];
}

// ==================== MAIN CONTENT BUFFER ====================

ob_start();

// ==================== PAGE ROUTING ====================

switch ($page) {
    
    // ==================== DASHBOARD ====================
    case 'dashboard':
        include __DIR__ . '/pages/dashboard.php';
        break;
    
    // ==================== PEOPLE MANAGEMENT ====================
    case 'people':
        include __DIR__ . '/pages/people.php';
        break;
    
    // ==================== PROFILE ====================
    case 'profile':
        include __DIR__ . '/pages/profile.php';
        break;
    
    // ==================== IMPORT ====================
    case 'import':
        requireRole(['admin', 'editor']);
        include __DIR__ . '/pages/import.php';
        break;
    
    // ==================== MERGE QUEUE ====================
    case 'merges':
        requireRole(['admin', 'editor']);
        include __DIR__ . '/pages/merges.php';
        break;
    
    // ==================== REPORTS ====================
    case 'reports':
        include __DIR__ . '/pages/reports.php';
        break;
    
    // ==================== DOSE SHARING ====================
    case 'dose_sharing':
        include __DIR__ . '/pages/dose_sharing.php';
        break;
    
    // ==================== SETTINGS ====================
    case 'settings':
        requireRole('admin');
        include __DIR__ . '/pages/settings.php';
        break;
    
    // ==================== AUDIT LOG ====================
    case 'audit':
        requireRole('admin');
        include __DIR__ . '/pages/audit.php';
        break;
    
    // ==================== DEFAULT (404) ====================
    default:
        echo "<div class='card'>";
        echo "<h1>Page Not Found</h1>";
        echo "<p>The page you requested could not be found.</p>";
        echo "<a href='?page=dashboard' class='btn btn-primary'>Go to Dashboard</a>";
        echo "</div>";
}

$content = ob_get_clean();

// ==================== RENDER LAYOUT ====================

$theme = $settings['theme'] ?? 'dark';
?>
<!DOCTYPE html>
<html lang="en" data-theme="<?= $theme ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= esc($settings['site_title']) ?></title>
    <link rel="stylesheet" href="assets/style.css">
</head>
<body>
    <div class="header">
        <div class="header-content">
            <div class="logo"><?= esc($settings['site_title']) ?></div>
            <nav class="nav">
                <a href="?page=dashboard">Dashboard</a>
                <a href="?page=people">People</a>
                
                <?php if (in_array($_SESSION['role'], ['admin', 'editor'])): ?>
                <a href="?page=import">Import</a>
                <a href="?page=merges">Merges</a>
                <?php endif; ?>
                
                <a href="?page=reports">Reports</a>
                
                <?php foreach ($activeModules as $mod): ?>
                    <?php if ($mod['menu_title']): ?>
                        <a href="?page=<?= esc($mod['slug']) ?>">
                            <?= esc($mod['menu_icon']) ?> <?= esc($mod['menu_title']) ?>
                        </a>
                    <?php endif; ?>
                <?php endforeach; ?>
                
                <?php if ($_SESSION['role'] === 'admin'): ?>
                <a href="?page=settings">Settings</a>
                <a href="?page=audit">Audit</a>
                <?php endif; ?>
                
                <span class="user-info"><?= esc($_SESSION['username']) ?></span>
                <a href="?page=logout">Logout</a>
            </nav>
        </div>
    </div>
    
    <div class="container">
        <?= flashMessage() ?>
        <?= $content ?>
    </div>
    
    <script src="assets/script.js"></script>
</body>
</html>
