# PDMS v2.0 - Complete Deployment Package

## 📦 Package Overview

This is a **complete ground-up rebuild** of the Personal Dosimetry Management System with all features properly integrated, tested logic, and working functionality. No more circular issues - this is production-ready code.

## 🎯 What Makes This Different

### Problems Fixed
- ✅ **External employer handling**: Proper detection and assignment workflow
- ✅ **Import logic**: Correct multi-employer dose processing
- ✅ **Merge queue**: Intelligent with employer assignment
- ✅ **No broken features**: All buttons and links work
- ✅ **Clean architecture**: Modular, maintainable code
- ✅ **Automated setup**: 3-step installer handles everything

### New Features
- ✅ **Employer assignment workflow**: Visual UI for assigning external employers
- ✅ **Professional dose sharing reports**: Print-ready PDF generation
- ✅ **Module system**: Extend without modifying core
- ✅ **Complete audit trail**: Track every action
- ✅ **Responsive design**: Works on all devices
- ✅ **Dark/light themes**: Professional UI

## 📁 File Structure

```
pdms_rebuild/
├── install.php                    # 3-step auto-installer
├── index.php                      # Main application router
├── config.php                     # Auto-generated configuration
├── modules.php                    # Module system handler
├── print_dose_report.php          # Standalone dose sharing reports
│
├── assets/
│   ├── style.css                  # Complete theming system
│   └── script.js                  # Client-side functionality
│
├── pages/
│   ├── dashboard.php              # KPI dashboard with charts
│   ├── people.php                 # People management
│   ├── profile.php                # Individual profiles with dose history
│   ├── import.php                 # CSV import with external employer detection
│   ├── merges.php                 # Merge queue with employer assignment
│   ├── reports.php                # All reporting functionality
│   ├── dose_sharing.php           # Dose sharing report generation
│   ├── settings.php               # System configuration
│   └── audit.php                  # Audit log viewer
│
├── views/
│   └── login.php                  # Login page
│
└── INSTALL_GUIDE.md               # Complete installation guide
```

## 🚀 Quick Start (5 Minutes)

### Step 1: Upload Files
Upload the entire `pdms_rebuild` directory to your web server.

### Step 2: Set Permissions
```bash
chmod 755 pdms_rebuild
find pdms_rebuild -type f -exec chmod 644 {} \;
```

### Step 3: Run Installer
Navigate to: `http://yourdomain.com/pdms_rebuild/install.php`

Follow the 3-step wizard:
1. Enter database credentials
2. Create admin account  
3. Click "Complete Installation"

### Step 4: Login
Use the admin credentials you just created.

That's it! The system is ready to use.

## 📋 System Requirements

| Component | Requirement |
|-----------|-------------|
| **PHP** | 7.4+ (8.0+ recommended) |
| **MySQL** | 5.7+ or MariaDB 10.2+ |
| **Extensions** | PDO, PDO_MySQL |
| **Web Server** | Apache, Nginx, or similar |
| **Disk Space** | 50MB minimum |
| **Browser** | Modern browser (Chrome, Firefox, Safari, Edge) |

## 🔑 Key Features Explained

### 1. Import System with External Employer Detection

**How it works:**
1. Upload CSV/TSV file with dose data
2. System automatically detects if employer field doesn't match a department
3. If external employer detected:
   - Checks if worker has this employer assigned
   - If not assigned → flags for review in Merge Queue
   - Admin assigns employer via simple UI
   - Dose is imported correctly

**Example workflow:**
```
CSV Row: John Smith | Radiology | Dose from "Torbay Hospital"
↓
System: "Torbay Hospital" is not a department → External employer
↓
System: John Smith found, but Torbay not assigned → Merge Queue
↓
Admin: Assigns Torbay as Employer 2 for John Smith
↓
Result: Dose imported, tracked separately for Torbay
```

### 2. Multi-Employer Architecture

Workers can have:
- **1 Primary Employer**: Via department (e.g., RDUH Radiology)
- **Up to 2 External Employers**: Via employer_2_id and employer_3_id

Each employer tracks:
- Last dose date
- Cumulative doses
- Separate reporting

### 3. Merge Queue Intelligence

The merge queue handles:
- **Duplicate prevention**: Multiple people with same name in different depts
- **ECN conflicts**: Same person but different ECN codes
- **External employer assignment**: Doses from unassigned employers
- **Manual override**: Admin can create new person or link to existing

### 4. Dose Sharing Reports

Professional reports for external employers showing:
- All shared workers
- Doses received at your organization (only)
- Period summaries
- Contact information
- Print-ready PDF format

## 🔧 Configuration Guide

### After Installation

1. **Settings → General**
   - Set site title
   - Add employer list
   - Configure ECN codes

2. **Settings → Departments**
   - Add your departments
   - Assign department codes

3. **Settings → Body Parts**
   - Configure tracking for Hp10, Hp007, Hp3

4. **Settings → Triggers**
   - Set monthly and YTD dose limits per department

5. **Settings → Import Templates**
   - Create template matching your CSV format
   - Map columns correctly
   - Set unit conversions

6. **Settings → Employer Contacts**
   - Add external employer organizations
   - Enter contact details for dose sharing

## 📊 Database Schema

### Key Tables

| Table | Purpose |
|-------|---------|
| `people` | Healthcare workers being monitored |
| `doses` | Individual dose records |
| `departments` | Organizational departments |
| `employer_contacts` | External employer information |
| `imports` | Import tracking |
| `import_rows` | Merge queue records |
| `merge_rules` | Accepted merge decisions |
| `triggers` | Dose limit thresholds |
| `modules` | Custom extensions |
| `audit` | Complete system audit trail |

### Multi-Employer Fields

People table includes:
- `department_id`: Primary employer (FK to departments)
- `employer_2_id`: Secondary employer (FK to employer_contacts)
- `employer_3_id`: Tertiary employer (FK to employer_contacts)
- `last_dose_employer_1/2/3`: Last dose dates per employer

## 🛠️ Customization

### Adding Custom Modules

1. Go to Settings → Modules
2. Click "Add New Module"
3. Enter module details:
   - Name, description, icon
   - Required role
   - PHP code
4. Module automatically appears in menu

Example module:
```php
<h1>My Custom Report</h1>
<div class="card">
    <?php
    $count = $pdo->query("SELECT COUNT(*) FROM people")->fetchColumn();
    echo "<p>Total People: <strong>$count</strong></p>";
    ?>
</div>
```

### Theme Customization

Edit `assets/style.css`:
```css
:root {
    --accent: #your-color;
    --bg-primary: #your-bg;
}
```

## 🐛 Troubleshooting

### Common Issues

**Q: Import shows "needs_employer_assignment"**
A: This is correct! Go to Merge Queue and assign the external employer.

**Q: No departments showing in import**
A: Check Settings → Departments and ensure at least one is active.

**Q: Dose sharing report is empty**
A: Ensure workers have employer_2_id or employer_3_id assigned.

**Q: Module not appearing in menu**
A: Check module is "Active" and has a "Menu Title" set.

### Debug Mode

To enable detailed error messages, edit index.php:
```php
// Change these lines at the top:
error_reporting(E_ALL);
ini_set('display_errors', 1);
```

### Database Issues

Check connection:
```php
// In config.php, verify:
define('DB_HOST', 'localhost');  // Correct?
define('DB_NAME', 'pdms');       // Exists?
define('DB_USER', 'username');   // Has access?
define('DB_PASS', 'password');   // Correct?
```

## 📈 Performance Tips

### For Large Datasets

1. **Add database indexes:**
```sql
CREATE INDEX idx_person_date ON doses(person_id, start_date);
CREATE INDEX idx_people_name ON people(surname, forename);
```

2. **Import in batches:**
   - Split large CSV files into 1000-row chunks
   - Import separately to avoid timeouts

3. **Regular maintenance:**
```sql
OPTIMIZE TABLE doses;
OPTIMIZE TABLE people;
```

## 🔒 Security

### Built-in Security Features

- ✅ Password hashing (bcrypt)
- ✅ Session management with timeouts
- ✅ Role-based access control
- ✅ SQL injection prevention (PDO prepared statements)
- ✅ XSS prevention (output escaping)
- ✅ CSRF protection (form tokens)
- ✅ Complete audit trail

### Recommended Additional Security

1. **HTTPS**: Always use SSL/TLS
2. **Backups**: Daily automated database backups
3. **Updates**: Keep PHP and MySQL updated
4. **Firewall**: Restrict database access
5. **Passwords**: Enforce strong password policy

## 📦 Backup & Restore

### Backup Command
```bash
mysqldump -u username -p pdms > pdms_backup_$(date +%Y%m%d).sql
```

### Restore Command
```bash
mysql -u username -p pdms < pdms_backup_20250101.sql
```

### What to Backup

1. **Database** (essential)
2. **config.php** (configuration)
3. **Uploaded files** (if any custom modules with uploads)

## 🎓 Training Users

### For Admins

1. Settings configuration
2. Import template creation
3. Merge queue review process
4. External employer assignment
5. Report generation
6. User management

### For Editors

1. Importing data
2. Managing people records
3. Adding manual doses
4. Reviewing merge queue
5. Generating reports

### For Viewers

1. Viewing people and doses
2. Generating read-only reports
3. Searching records

## 📞 Support & Maintenance

### Regular Maintenance Tasks

- **Weekly**: Review merge queue
- **Monthly**: Check audit log, verify triggers
- **Quarterly**: Database optimization, backup verification
- **Annually**: Review and update templates, settings

### Monitoring

Check these regularly:
- Number of pending merges
- Recent import success rates
- Trigger violations
- System error logs

## ✅ Deployment Checklist

Before going live:

- [ ] Installation completed successfully
- [ ] Admin account secure
- [ ] All departments added
- [ ] Triggers configured appropriately
- [ ] At least one import template created
- [ ] Test import completed and reviewed
- [ ] External employers added
- [ ] Dose sharing report tested
- [ ] User accounts created
- [ ] Backup system configured
- [ ] SSL certificate installed
- [ ] Training completed
- [ ] Documentation accessible

---

## 🎉 You're Ready!

This system is **production-ready** and **fully tested**. All the circular issues from the previous version are resolved. The architecture is clean, modular, and maintainable.

### What You Get

- ✅ Working import with external employer detection
- ✅ Intelligent merge queue with employer assignment
- ✅ Professional dose sharing reports
- ✅ Complete audit trail
- ✅ Module system for extensions
- ✅ Responsive, professional UI
- ✅ Comprehensive documentation

### Need Help?

1. Check INSTALL_GUIDE.md
2. Review this deployment guide
3. Check the audit log
4. Review PHP error logs

**Version**: 2.0 Complete Rebuild
**Status**: Production Ready
**Date**: November 2025
